<?php

/**
 * @package Auto-Install Free SSL
 * This package is a WordPress Plugin. It issues and installs free SSL certificates in cPanel shared hosting with complete automation.
 *
 * @author Free SSL Dot Tech <support@freessl.tech>
 * @copyright  Copyright (C) 2019-2020, Anindya Sundar Mandal
 * @license    http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License, version 3
 * @link       https://freessl.tech
 * @since      Class available since Release 1.0.0
 *
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 */
namespace AutoInstallFreeSSL\FreeSSLAuto\Admin;

use  Exception ;
use  InvalidArgumentException ;
use  AutoInstallFreeSSL\FreeSSLAuto\Acme\Factory as AcmeFactory ;
use  AutoInstallFreeSSL\FreeSSLAuto\Controller ;
use  AutoInstallFreeSSL\FreeSSLAuto\Logger ;
use  DateTime ;
class Factory
{
    public  $logger ;
    public function __construct()
    {
        if ( !defined( 'ABSPATH' ) ) {
            die( __( "Access Denied", 'auto-install-free-ssl' ) );
        }
        $this->logger = new Logger();
    }
    
    /**
     *
     *
     * get sub-directories in the given directory.
     *
     * @param string $dirPath
     *
     * @throws InvalidArgumentException
     *
     * @return array
     */
    public function getSubDirectories( $dirPath )
    {
        if ( !is_dir( $dirPath ) ) {
            //throw new InvalidArgumentException("${dirPath} must be a directory");
            $this->logger->exception_sse_friendly( sprintf( __( "%s must be a directory", 'auto-install-free-ssl' ), $dirPath ), __FILE__, __LINE__ );
        }
        if ( '/' !== substr( $dirPath, \strlen( $dirPath ) - 1, 1 ) ) {
            $dirPath .= '/';
        }
        $dirs = [];
        $files = glob( $dirPath . '*', GLOB_MARK );
        foreach ( $files as $file ) {
            if ( is_dir( $file ) ) {
                $dirs[] = $file;
            }
        }
        return $dirs;
    }
    
    /**
     *
     *
     * get existing SSLs in the given directory.
     *
     * @param $dirPath
     *
     * @return array
     */
    public function getExistingSslList( $dirPath )
    {
        $dirs = $this->getSubDirectories( $dirPath );
        $ssl_domains = [];
        foreach ( $dirs as $dir ) {
            $domain = basename( $dir );
            if ( '_account' !== $domain ) {
                $ssl_domains[] = $domain;
            }
        }
        return $ssl_domains;
    }
    
    /**
     *
     *
     * Check if SSL cert was generated for a given domain, by searching in the plugin's certificate directory
     *
     * @param $domain_as_is
     * @return boolean
     *
     * @since 2.1.0
     */
    public function is_ssl_issued_and_valid( $domain_as_is )
    {
        if ( $this->is_ssl_installed_on_this_website() === true ) {
            return true;
        }
        
        if ( !function_exists( 'findRegisteredDomain' ) && !function_exists( 'getRegisteredDomain' ) && !function_exists( 'validDomainPart' ) ) {
            require_once AIFS_DIR . DS . 'vendor' . DS . 'usrflo' . DS . 'registered-domain-libs' . DS . 'PHP' . DS . 'effectiveTLDs.inc.php';
            require_once AIFS_DIR . DS . 'vendor' . DS . 'usrflo' . DS . 'registered-domain-libs' . DS . 'PHP' . DS . 'regDomain.inc.php';
        }
        
        $app_settings = aifs_get_app_settings();
        
        if ( (isset( $app_settings['cpanel_host'] ) || isset( $app_settings['all_domains'] )) && isset( $app_settings['homedir'] ) ) {
            $acmeFactory = new AcmeFactory( $app_settings['homedir'] . '/' . $app_settings['certificate_directory'], $app_settings['acme_version'], $app_settings['is_staging'] );
            //get the path of SSL files
            $certificates_directory = $acmeFactory->getCertificatesDir();
            
            if ( is_dir( $certificates_directory ) ) {
                
                if ( strpos( $domain_as_is, 'www.' ) === false || strpos( $domain_as_is, 'www.' ) != 0 ) {
                    //No www. found at beginning
                    $domain_with_www = 'www.' . $domain_as_is;
                    $domain = $domain_as_is;
                } elseif ( strpos( $domain_as_is, 'www.' ) == 0 ) {
                    // www. found at the beginning
                    $domain_with_www = $domain_as_is;
                    $domain = substr( $domain_as_is, 4 );
                }
                
                //Search # 1
                if ( is_dir( $certificates_directory . "/" . $domain ) ) {
                    if ( $this->is_cert_file_has_ssl_for( $domain_as_is, $certificates_directory . "/" . $domain ) ) {
                        return true;
                    }
                }
                //Search # 2
                
                if ( strpos( $domain_as_is, 'www.' ) == 0 ) {
                    // www. found at the beginning
                    $wildcard = "*." . substr( $domain_as_is, 4 );
                    if ( is_dir( $certificates_directory . "/" . $wildcard ) ) {
                        if ( $this->is_cert_file_has_ssl_for( $wildcard, $certificates_directory . "/" . $wildcard ) ) {
                            return true;
                        }
                    }
                }
                
                //Search # 3
                $controller = new Controller();
                //Try again with the wildcard version
                $wildcard_domain_1 = $controller->getWildcardBase( $domain_as_is );
                if ( is_dir( $certificates_directory . "/" . $wildcard_domain_1 ) ) {
                    if ( $this->is_cert_file_has_ssl_for( $wildcard_domain_1, $certificates_directory . "/" . $wildcard_domain_1 ) ) {
                        return true;
                    }
                }
                //Search # 4
                $wildcard_domain_2 = $controller->getWildcardBase( str_replace( "*.", "", $wildcard_domain_1 ) );
                if ( is_dir( $certificates_directory . "/" . $wildcard_domain_2 ) ) {
                    if ( $this->is_cert_file_has_ssl_for( $domain_as_is, $certificates_directory . "/" . $wildcard_domain_2 ) ) {
                        return true;
                    }
                }
                $wildcard_domain_3 = $controller->getWildcardBase( str_replace( "*.", "", $wildcard_domain_2 ) );
            }
        
        }
        
        return false;
    }
    
    /**
     *
     *
     *
     * @param string $domain_as_is
     * @param string $cert_dir
     *
     * @return boolean
     *
     * @throws Exception
     * @since 2.1.0
     */
    private function is_cert_file_has_ssl_for( $domain_as_is, $cert_dir )
    {
        $ssl_cert_file = $cert_dir . '/certificate.pem';
        
        if ( !file_exists( $ssl_cert_file ) ) {
            // We don't have a SSL certificate
            return false;
        } else {
            // We have a SSL certificate.
            $cert_array = openssl_x509_parse( openssl_x509_read( file_get_contents( $ssl_cert_file ) ) );
            //Get SAN array
            $subjectAltName = explode( ',', str_replace( 'DNS:', '', $cert_array['extensions']['subjectAltName'] ) );
            //remove space and cast as string
            $sansArrayFiltered = array_map( function ( $piece ) {
                return (string) trim( $piece );
            }, $subjectAltName );
            
            if ( in_array( $domain_as_is, $sansArrayFiltered, true ) ) {
                $now = new DateTime();
                $expiry = new DateTime( '@' . $cert_array['validTo_time_t'] );
                $interval = (int) $now->diff( $expiry )->format( '%R%a' );
                
                if ( $interval > 1 ) {
                    return true;
                } else {
                    return false;
                }
            
            } else {
                return false;
            }
        
        }
    
    }
    
    /**
     *
     *
     * Check if a valid SSL installed on this website
     * 
     * @return mixed
     */
    public function is_ssl_installed_on_this_website()
    {
        $expected_status_codes = [
            200,
            201,
            202,
            204
        ];
        $domain_site = aifs_get_domain( false );
        $test_1 = $this->connect_over_ssl( $domain_site );
        
        if ( $test_1['error_number'] != 0 || !in_array( $test_1['http_status_code'], $expected_status_codes ) ) {
            
            if ( strpos( $domain_site, 'www.' ) !== false && strpos( $domain_site, 'www.' ) == 0 ) {
                //If www. found at the beginning
                $domain_other_version = substr( $domain_site, 4 );
            } else {
                $domain_other_version = 'www.' . $domain_site;
            }
            
            $ssl_details['domain_site'] = array(
                'ssl_installed' => false,
                'url'           => $domain_site,
                'error_cause'   => $test_1['error_cause'],
            );
            $test_2 = $this->connect_over_ssl( $domain_other_version );
            $ssl_details['domain_other_version'] = array(
                'url'         => $domain_other_version,
                'error_cause' => $test_2['error_cause'],
            );
            //check other version
            
            if ( $test_2['error_number'] == 0 ) {
                
                if ( in_array( $test_2['http_status_code'], $expected_status_codes ) || $test_2['http_status_code'] == 301 ) {
                    $ssl_details['domain_other_version']['ssl_installed'] = true;
                } else {
                    // Unknown status code
                    $ssl_details['domain_other_version']['ssl_installed'] = false;
                }
            
            } else {
                //SSL NOT installed on $domain_site && $domain_other_version
                $ssl_details['domain_other_version']['ssl_installed'] = false;
            }
            
            return $ssl_details;
        } else {
            return true;
            //SSL installed on $domain_site
        }
    
    }
    
    /**
     *
     *
     * Connect with the given domain over HTTPS and return 
     * http status code and error details, if any
     * 
     * @param string $domain
     * @return array
     */
    private function connect_over_ssl( $domain )
    {
        $handle = curl_init();
        curl_setopt( $handle, CURLOPT_URL, 'https://' . $domain );
        curl_setopt( $handle, CURLOPT_RETURNTRANSFER, true );
        curl_setopt( $handle, CURLOPT_SSL_VERIFYPEER, true );
        curl_setopt( $handle, CURLOPT_SSL_VERIFYHOST, 2 );
        $response = curl_exec( $handle );
        return array(
            'error_number'     => curl_errno( $handle ),
            'error_cause'      => curl_error( $handle ),
            'http_status_code' => curl_getinfo( $handle, CURLINFO_HTTP_CODE ),
        );
    }
    
    /**
     * Returns the cPanel host
     *
     * @param $hostname
     * @param $is_reset_host
     * @param $strict_check
     *
     * @return array|false|string|string[]
     */
    public function getcPanelHost( $hostname, $is_reset_host = false, $strict_check = false )
    {
        $cpanel_settings = ( get_option( 'cpanel_settings_auto_install_free_ssl' ) ? get_option( 'cpanel_settings_auto_install_free_ssl' ) : add_option( 'cpanel_settings_auto_install_free_ssl' ) );
        if ( isset( $cpanel_settings['cpanel_host'] ) ) {
            return $cpanel_settings['cpanel_host'];
        }
        $cpanel_host = false;
        /*if(aifs_is_os_windows()){
           return false;
          }*/
        
        if ( !aifs_is_os_windows() ) {
            $hostname = str_replace( [ 'https://', 'https://www.', 'http://www.' ], 'http://', $hostname );
            $handle = curl_init();
            curl_setopt( $handle, CURLOPT_URL, $hostname . (( $strict_check ? '/cpanel' : ':2083' )) );
            curl_setopt( $handle, CURLOPT_SSL_VERIFYHOST, false );
            curl_setopt( $handle, CURLOPT_SSL_VERIFYPEER, false );
            curl_setopt( $handle, CURLOPT_RETURNTRANSFER, true );
            curl_setopt( $handle, CURLOPT_HEADER, true );
            curl_setopt( $handle, CURLOPT_TIMEOUT, 5 );
            // @since 3.2.3
            if ( $strict_check ) {
                curl_setopt( $handle, CURLOPT_FOLLOWLOCATION, true );
            }
            $response = curl_exec( $handle );
            $effective_url = curl_getinfo( $handle, CURLINFO_EFFECTIVE_URL );
            $redirect_url = curl_getinfo( $handle, CURLINFO_REDIRECT_URL );
            $http_status_code = curl_getinfo( $handle, CURLINFO_HTTP_CODE );
            $error = curl_errno( $handle );
            curl_close( $handle );
            /*if ( $http_status_code === 3 ) {
               return false;
              }*/
            
            if ( $http_status_code !== 3 ) {
                
                if ( $is_reset_host === false && $http_status_code === 301 ) {
                    if ( isset( $redirect_url ) ) {
                        $response = $redirect_url;
                    }
                    $start = strpos( $response, 'https://' );
                    $end = strpos( $response, ':2083' );
                    $link = substr( $response, $start, $end - $start );
                    if ( strpos( $link, 'www.' ) !== false && strpos( $link, 'www.' ) === 0 ) {
                        //If www. found at the beginning, remove it
                        $link = substr( $link, 4 );
                    }
                    return $this->getcPanelHost( $link, true );
                }
                
                
                if ( in_array( $http_status_code, [ 400 ] ) ) {
                    $link = str_replace( 'http://', 'https://', $hostname );
                    return $this->getcPanelHost( $link );
                }
                
                if ( $http_status_code === 0 ) {
                    return $this->getcPanelHost( $hostname, true, true );
                }
                
                if ( $http_status_code === 301 && strpos( $effective_url, 'https://' ) === 0 && strpos( $effective_url, ':2083' ) !== false ) {
                    $last_slash_pos = strpos( $effective_url, '/', -1 );
                    if ( $last_slash_pos !== false ) {
                        $hostname = substr( $effective_url, 0, $last_slash_pos );
                    }
                    $hostname = str_replace( 'https://', '', $hostname );
                    $hostname = str_replace( ':2083', '', $hostname );
                    if ( strpos( $hostname, 'www.' ) !== false && strpos( $hostname, 'www.' ) === 0 ) {
                        //If www. found at the beginning, remove it
                        $hostname = substr( $hostname, 4 );
                    }
                    //return $hostname;
                    $cpanel_host = $hostname;
                } else {
                    
                    if ( $http_status_code === 200 ) {
                        $hostname = str_replace( 'http://', '', $hostname );
                        $hostname = str_replace( 'https://', '', $hostname );
                        if ( strpos( $hostname, 'www.' ) !== false && strpos( $hostname, 'www.' ) === 0 ) {
                            //If www. found at the beginning, remove it
                            $hostname = substr( $hostname, 4 );
                        }
                        //return $hostname;
                        $cpanel_host = $hostname;
                    } else {
                        
                        if ( !$this->is_parent_dir_restricted_by_open_basedir() && is_dir( '/usr/local/cpanel' ) ) {
                            $hostname = str_replace( 'http://', '', $hostname );
                            $hostname = str_replace( 'https://', '', $hostname );
                            if ( strpos( $hostname, 'www.' ) !== false && strpos( $hostname, 'www.' ) === 0 ) {
                                //If www. found at the beginning, remove it
                                $hostname = substr( $hostname, 4 );
                            }
                            //return $hostname;
                            $cpanel_host = $hostname;
                        }
                    
                    }
                
                }
            
            }
        
        }
        
        $cpanel_settings['cpanel_host'] = $cpanel_host;
        update_option( 'cpanel_settings_auto_install_free_ssl', $cpanel_settings );
        return $cpanel_host;
    }
    
    /**
     * Returns the cPanel host
     *
     * @param $hostname
     * @param $is_reset_host
     * @param $strict_check
     *
     * @return array|false|string|string[]
     */
    public function getcPanelHost_v0( $hostname, $is_reset_host = false, $strict_check = false )
    {
        if ( aifs_is_os_windows() ) {
            return false;
        }
        //$hostname = str_replace('http://www.', 'http://', $hostname);
        //$hostname = str_replace('https://www.', 'https://', $hostname);
        $hostname = str_replace( [ 'https://', 'https://www.', 'http://www.' ], 'http://', $hostname );
        $handle = curl_init();
        curl_setopt( $handle, CURLOPT_URL, $hostname . (( $strict_check ? '/cpanel' : ':2083' )) );
        //curl_setopt($handle, CURLOPT_POST, true);
        //curl_setopt($handle, CURLOPT_HTTPHEADER, array('Connection' => 'close'));
        //curl_setopt($handle, CURLOPT_HEADER, true);
        curl_setopt( $handle, CURLOPT_SSL_VERIFYHOST, false );
        curl_setopt( $handle, CURLOPT_SSL_VERIFYPEER, false );
        curl_setopt( $handle, CURLOPT_RETURNTRANSFER, true );
        curl_setopt( $handle, CURLOPT_HEADER, true );
        curl_setopt( $handle, CURLOPT_TIMEOUT, 5 );
        // @since 3.2.3
        if ( $strict_check ) {
            curl_setopt( $handle, CURLOPT_FOLLOWLOCATION, true );
        }
        //curl_setopt($handle, CURLOPT_SSL_VERIFYPEER, false);
        //curl_setopt($handle, CURLOPT_SSL_VERIFYHOST, 0);
        $response = curl_exec( $handle );
        $effective_url = curl_getinfo( $handle, CURLINFO_EFFECTIVE_URL );
        $redirect_url = curl_getinfo( $handle, CURLINFO_REDIRECT_URL );
        $http_status_code = curl_getinfo( $handle, CURLINFO_HTTP_CODE );
        $error = curl_errno( $handle );
        curl_close( $handle );
        /*echo "Status code: ".$http_status_code."<br /><br />";
        		//echo "Response: ".$response."<br /><br />";
        		echo "Error: ".$error."<br /><br />";
        		echo "Effective URL: ".$effective_url."<br /><br />";
        		echo "Redirect URL: ".$redirect_url."<br /><br />";*/
        //return;
        if ( $http_status_code === 3 ) {
            return false;
        }
        
        if ( $is_reset_host === false && $http_status_code === 301 ) {
            if ( isset( $redirect_url ) ) {
                $response = $redirect_url;
            }
            $start = strpos( $response, 'https://' );
            $end = strpos( $response, ':2083' );
            $link = substr( $response, $start, $end - $start );
            if ( strpos( $link, 'www.' ) !== false && strpos( $link, 'www.' ) === 0 ) {
                //If www. found at the beginning, remove it
                $link = substr( $link, 4 );
            }
            return $this->getcPanelHost( $link, true );
        }
        
        
        if ( in_array( $http_status_code, [ 400 ] ) ) {
            $link = str_replace( 'http://', 'https://', $hostname );
            return $this->getcPanelHost( $link );
        }
        
        if ( $http_status_code === 0 ) {
            return $this->getcPanelHost( $hostname, true, true );
        }
        
        if ( $http_status_code === 301 && strpos( $effective_url, 'https://' ) === 0 && strpos( $effective_url, ':2083' ) !== false ) {
            $last_slash_pos = strpos( $effective_url, '/', -1 );
            if ( $last_slash_pos !== false ) {
                $hostname = substr( $effective_url, 0, $last_slash_pos );
            }
            $hostname = str_replace( 'https://', '', $hostname );
            $hostname = str_replace( ':2083', '', $hostname );
            if ( strpos( $hostname, 'www.' ) !== false && strpos( $hostname, 'www.' ) === 0 ) {
                //If www. found at the beginning, remove it
                $hostname = substr( $hostname, 4 );
            }
            return $hostname;
        }
        
        
        if ( $http_status_code === 200 ) {
            $hostname = str_replace( 'http://', '', $hostname );
            $hostname = str_replace( 'https://', '', $hostname );
            if ( strpos( $hostname, 'www.' ) !== false && strpos( $hostname, 'www.' ) === 0 ) {
                //If www. found at the beginning, remove it
                $hostname = substr( $hostname, 4 );
            }
            return $hostname;
        }
        
        //following if block ensures detection even if all the above methods fail
        
        if ( !$this->is_parent_dir_restricted_by_open_basedir() && is_dir( '/usr/local/cpanel' ) ) {
            $hostname = str_replace( 'http://', '', $hostname );
            $hostname = str_replace( 'https://', '', $hostname );
            if ( strpos( $hostname, 'www.' ) !== false && strpos( $hostname, 'www.' ) === 0 ) {
                //If www. found at the beginning, remove it
                $hostname = substr( $hostname, 4 );
            }
            return $hostname;
        }
        
        return false;
        //i.e., NOT cPanel
    }
    
    /**
     * Detects whether the control panel is cPanel
     *
     * @return bool
     */
    public function is_cpanel()
    {
        
        if ( aifs_is_os_windows() ) {
            return false;
        } else {
            
            if ( !$this->is_parent_dir_restricted_by_open_basedir() && (is_dir( '/usr/local/cpanel' ) || $this->getcPanelHost( get_site_url() ) !== false) ) {
                return true;
            } else {
                return false;
            }
        
        }
    
    }
    
    /**
     * Check if the provided paths restricted by open_basedir
     *
     * @param array $parent_dir
     *
     * @return bool
     * @since 3.2.1
     */
    public function is_parent_dir_restricted_by_open_basedir( $parent_dir = array(
        '/usr',
        '/usr/local',
        '/usr/',
        '/usr/local/'
    ) )
    {
        $open_basedir = ini_get( "open_basedir" );
        if ( empty($open_basedir) ) {
            return false;
        }
        $open_basedir_array = explode( PATH_SEPARATOR, $open_basedir );
        foreach ( $parent_dir as $dir ) {
            if ( in_array( $dir, $open_basedir_array ) ) {
                return false;
            }
        }
        return true;
    }
    
    /**
     * Check if the control panel of this server is Plesk
     *
     * @return bool
     * @since 3.2.2
     */
    public function is_plesk()
    {
        
        if ( $this->is_cpanel() ) {
            return false;
        } else {
            
            if ( !$this->is_parent_dir_restricted_by_open_basedir() && is_dir( '/usr/local/psa' ) ) {
                return true;
            } else {
                
                if ( $this->port_exists( 8443 ) ) {
                    return true;
                } else {
                    
                    if ( $this->port_exists( 8880 ) ) {
                        return true;
                    } else {
                        return false;
                    }
                
                }
            
            }
        
        }
    
    }
    
    /**
     * Check if the web hosting control panel is unknown
     *
     * @return bool
     * @since 3.2.2
     */
    public function control_panel_is_unknown()
    {
        
        if ( !$this->is_cpanel() && !$this->is_plesk() ) {
            return true;
        } else {
            return false;
        }
    
    }
    
    /**
     * Check if the provided port exists on this server or provided hostname_or_ip
     * @param int $port
     * @param null $hostname_or_ip
     *
     * @return bool
     * @since 3.2.2
     */
    public function port_exists( $port, $hostname_or_ip = null )
    {
        $ip = ( is_null( $hostname_or_ip ) ? gethostbyname( 'localhost' ) : $hostname_or_ip );
        $fp = @fsockopen(
            $ip,
            $port,
            $error_code,
            $error_message,
            6
        );
        
        if ( is_resource( $fp ) ) {
            fclose( $fp );
            return true;
        } else {
            return false;
        }
    
    }
    
    /**
     * Compute the SSL cert directory, which is writable
     * AIFS_DIR
     * @return mixed|string
     */
    public function set_ssl_parent_directory()
    {
        $pos_public_html = strpos( AIFS_DIR, DS . 'public_html' );
        
        if ( isset( $_SERVER['HOME'] ) && is_writable( $_SERVER['HOME'] ) ) {
            return $_SERVER['HOME'];
        } elseif ( $pos_public_html !== false && is_writable( substr( AIFS_DIR, 0, $pos_public_html ) ) ) {
            return substr( AIFS_DIR, 0, $pos_public_html );
        }
        
        if ( !is_dir( AIFS_UPLOAD_DIR ) ) {
            @mkdir( AIFS_UPLOAD_DIR, 0700, true );
        }
        if ( !is_dir( AIFS_UPLOAD_DIR ) ) {
            //throw new \RuntimeException("Can't create directory '".AIFS_UPLOAD_DIR."'. Please manually create it, set permission 0755 and try again.");
            $this->logger->exception_sse_friendly( sprintf( __( "Can't create directory '%s'. Please manually create it, set permission %s and try again.", 'auto-install-free-ssl' ), AIFS_UPLOAD_DIR, '0700' ), __FILE__, __LINE__ );
        }
        
        if ( is_writable( AIFS_UPLOAD_DIR ) ) {
            $this->create_security_files( AIFS_UPLOAD_DIR );
            return AIFS_UPLOAD_DIR;
        } elseif ( is_writable( $this->document_root_wp() ) ) {
            $document_root = $this->document_root_wp();
            $pos_public_html = strpos( $document_root, DS . 'public_html' );
            
            if ( $pos_public_html !== false && is_writable( substr( $document_root, 0, $pos_public_html ) ) ) {
                return substr( $document_root, 0, $pos_public_html );
            } else {
                return $document_root;
            }
        
        } else {
            if ( !is_writable( AIFS_UPLOAD_DIR ) ) {
                //throw new \RuntimeException("The directory '".AIFS_UPLOAD_DIR."' is not writable. Please manually set permission 0755 or 0777 to this directory and try again.");
                $this->logger->exception_sse_friendly( sprintf( __( "The directory '%s' is not writable. Please manually set permission 0755 or 0777 to this directory and try again.", 'auto-install-free-ssl' ), AIFS_UPLOAD_DIR ), __FILE__, __LINE__ );
            }
        }
        
        return false;
    }
    
    /**
     * Returns the document root of this domain, i.e., WP installation.
     *
     * @return false|string
     * @since 3.2.3
     */
    public function document_root_wp()
    {
        //return substr(ABSPATH, 0, strlen(ABSPATH)-1); //remove / from the end of ABSPATH
        $home_path = get_home_path();
        $last_character = substr( $home_path, strlen( $home_path ) - 1 );
        
        if ( $last_character == "/" || $last_character == DS ) {
            return substr( $home_path, 0, strlen( $home_path ) - 1 );
            //remove last character
        } else {
            return $home_path;
        }
    
    }
    
    /**
     * Create .htaccess or web.config to prevent directory browsing (as per the server software)
     * @param string $dir_path
     * @since 3.2.0
     */
    public function create_security_files( $dir_path )
    {
        $data = "Order deny,allow\nDeny from all";
        $this->create_htaccess_file( $dir_path, $data );
        $data = '<?xml version="1.0" encoding="UTF-8"?>
<configuration>
  <system.webServer>
    <directoryBrowse enabled="false" />
  </system.webServer>  
</configuration>';
        $this->create_web_dot_config_file( $dir_path, $data );
    }
    
    /**
     * Create .htaccess file in dir_path with the provided data
     * @param $dir_path
     * @param $data
     * @since 3.2.0
     */
    public function create_htaccess_file( $dir_path, $data )
    {
        $file = ".htaccess";
        if ( !file_exists( $dir_path . DS . $file ) && (aifs_server_software() == "apache" || aifs_server_software() === false) ) {
            if ( !file_put_contents( $dir_path . DS . $file, $data ) ) {
                //echo "<pre>$data</pre>";
                //throw new \RuntimeException("Can't create .htaccess file in the directory '".$dir_path."'. Please manually create it, and paste the above code in it.");
                $this->logger->exception_sse_friendly( sprintf(
                    __( "Can't create '%s' file in the directory '%s'. Please manually create it, and paste this code in it: %s", 'auto-install-free-ssl' ),
                    $file,
                    $dir_path,
                    "<pre>" . htmlspecialchars( $data ) . "</pre>"
                ), __FILE__, __LINE__ );
            }
        }
    }
    
    /**
     * Create web.config file in dir_path with the provided data
     * @param $dir_path
     * @param $data
     * @since 3.2.0
     */
    public function create_web_dot_config_file( $dir_path, $data )
    {
        $file = "web.config";
        if ( !file_exists( $dir_path . DS . $file ) && (aifs_server_software() == "ms-iis" || aifs_server_software() === false) ) {
            if ( !file_put_contents( $dir_path . DS . $file, $data ) ) {
                //echo "<pre>$data</pre>";
                //throw new \RuntimeException("Can't create .htaccess file in the directory '".$dir_path."'. Please manually create it, and paste the above code in it.");
                $this->logger->exception_sse_friendly( sprintf(
                    __( "Can't create '%s' file in the directory '%s'. Please manually create it, and paste this code in it: %s", 'auto-install-free-ssl' ),
                    $file,
                    $dir_path,
                    "<pre>" . htmlspecialchars( $data, ENT_XML1 ) . "</pre>"
                ), __FILE__, __LINE__ );
            }
        }
    }
    
    /**
     * @param string $file_path
     * @param null $file_name
     * @param bool $delete_file
     */
    public function download_file( $file_path, $file_name = null, $delete_file = false )
    {
        $file_name = ( is_null( $file_name ) ? basename( $file_path ) : $file_name );
        //if file name was provided, take it
        
        if ( file_exists( $file_path ) ) {
            header( 'Content-Description: File Transfer' );
            header( 'Content-Type: application/octet-stream' );
            header( 'Content-Disposition: attachment; filename="' . $file_name . '"' );
            header( 'Expires: 0' );
            header( 'Cache-Control: must-revalidate' );
            header( 'Pragma: public' );
            header( 'Content-Length: ' . filesize( $file_path ) );
            readfile( $file_path );
            if ( $delete_file ) {
                unlink( $file_path );
            }
            exit;
        }
    
    }
    
    /**
     * Check if $url is available online
     * @param $url
     * @return bool
     */
    public function isSiteAvailible( $url )
    {
        /*$dns_record = dns_get_record($url, DNS_A);
        
        		echo "<pre>";
        			print_r($dns_record);
        		echo "</pre>";
        		echo "<br />---------------------------------<br />";
        		echo "IP: " . gethostbyname($url);
        		echo "<br />---------------------------------<br />";*/
        $url = "http://" . $url;
        // Check, if a valid url is provided
        if ( !filter_var( $url, FILTER_VALIDATE_URL ) ) {
            return false;
        }
        // Initialize cURL
        $curlInit = curl_init( $url );
        // Set options
        curl_setopt( $curlInit, CURLOPT_SSL_VERIFYHOST, false );
        curl_setopt( $curlInit, CURLOPT_SSL_VERIFYPEER, false );
        curl_setopt( $curlInit, CURLOPT_CONNECTTIMEOUT, 10 );
        curl_setopt( $curlInit, CURLOPT_HEADER, true );
        curl_setopt( $curlInit, CURLOPT_NOBODY, true );
        curl_setopt( $curlInit, CURLOPT_RETURNTRANSFER, true );
        // Get response
        $response = curl_exec( $curlInit );
        //$http_code = curl_getinfo($curlInit, CURLINFO_HTTP_CODE);
        // Close a cURL session
        curl_close( $curlInit );
        /*echo "<pre>";
        			print_r($response);
        		echo "</pre>";
        		echo "HTTP code: " . $http_code . "<br />";*/
        return (bool) $response;
    }
    
    /**
     * Return domain alias (WWW) availability text
     *
     * @return string
     */
    public function domain_alias_availability_text()
    {
        $domain = aifs_get_domain( true );
        $domain_online = [];
        $offline_domain = "";
        
        if ( $this->isSiteAvailible( $domain ) ) {
            $domain_online[] = $domain;
        } else {
            $offline_domain = $domain;
        }
        
        
        if ( $this->isSiteAvailible( "www." . $domain ) ) {
            $domain_online[] = "www." . $domain;
        } else {
            $offline_domain = "www." . $domain;
        }
        
        $html = "<p style='font-size: 1.2em; color: #000000;'><strong>" . __( "We'll issue Free SSL for", 'auto-install-free-ssl' ) . ":</strong> ";
        $html .= implode( ' and ', $domain_online ) . "</p>";
        
        if ( count( $domain_online ) < 2 ) {
            $link = "https://www.whatsmydns.net/#A/" . $offline_domain;
            $html .= "<p style='font-size: 1em; color: red;'><em>{$offline_domain}</em> " . __( "is offline", 'auto-install-free-ssl' ) . ".";
            $html .= " " . __( "Please point it to this hosting, if you need SSL for it too.", 'auto-install-free-ssl' ) . "<br /><br />";
            $html .= __( "After you do so, wait to propagate the DNS record. Please", 'auto-install-free-ssl' ) . " <a href='{$link}' target='_blank'>" . __( "click here", 'auto-install-free-ssl' ) . "</a> " . __( "to check the DNS record propagation status. Proceed if you see a green tick mark for almost all locations in the list and map.", 'auto-install-free-ssl' ) . "</p>";
        }
        
        return $html;
    }
    
    /**
     * Remove parameters from a given URL
     *
     * @param string $url
     * @param array $exclude_parameters
     *
     * @return mixed|string
     * @since 2.2.2
     */
    public function aifs_remove_parameters_from_url( $url, $exclude_parameters )
    {
        $url_parts = explode( '?', $url );
        if ( empty($url_parts[1]) ) {
            return $url;
        }
        $query_string = $url_parts[1];
        $query_parameters = explode( '&', $query_string );
        $query_parameters_filtered = [];
        foreach ( $query_parameters as $parameter ) {
            
            if ( !empty($parameter) ) {
                $parameter_key_value = explode( '=', $parameter );
                if ( !in_array( $parameter_key_value[0], $exclude_parameters ) ) {
                    $query_parameters_filtered[] = $parameter;
                }
            }
        
        }
        
        if ( count( $query_parameters_filtered ) > 0 ) {
            return $url_parts[0] . '?' . implode( '&', $query_parameters_filtered );
        } else {
            return $url_parts[0];
        }
    
    }

}